<h1 align="center">AR Maintenance & Inspection Assistant - Backend</h1>

<p align="center">
  A backend service for a real-time AR application assisting technicians with machine maintenance and inspection, built with NestJS and MQTT.
</p>

<p align="center" style="display: flex; justify-content: center; align-items: center; gap: 20px">
  <img src="https://nodejs.org/static/images/logo.svg" width="80" alt="Node.js" />
  <img src="https://nestjs.com/img/logo-small.svg" width="80" alt="NestJS" />
  <img src="https://raw.githubusercontent.com/mqttjs/MQTT.js/137ee0e3940c1f01049a30248c70f24dc6e6f829/MQTT.js.png" width="80" alt="MQTT" />
</p>

---

## Description

This backend API powers the **AR M&I Assistant**, a proof-of-concept augmented reality application developed for a school project at Windesheim Zwolle. The application connects to an MQTT broker to fetch real-time machine data and serves it to a Unity-based AR frontend via Server-Sent Events (SSE) and REST APIs.

> [!NOTE]
> This project demonstrates dynamic topic subscription, real-time data streaming, and efficient machine-level data handling across multiple endpoints.

---

## Related Repositories

- [Frontend (Unity AR)](https://github.com/CSchouwenaar/AR-Maintance-and-Inspection-Assistant-Frontend)  
  Unity-based AR interface that visualizes real-time machine data retrieved from this backend.

---

## Tech Stack

| Tech                                            | Purpose                                                 |
| ----------------------------------------------- | ------------------------------------------------------- |
| [NestJS](https://nestjs.com)                    | Backend framework for modular and scalable architecture |
| [MQTT.js](https://github.com/mqttjs/MQTT.js)    | Lightweight messaging protocol for IoT communication    |
| [RxJS](https://rxjs.dev/)                       | Reactive programming with Observables                   |
| [Swagger](https://swagger.io/tools/swagger-ui/) | API documentation                                       |
| TypeScript                                      | Strong typing and clean structure                       |
| Node.js                                         | Runtime environment                                     |
| [Prisma.js](https://www.prisma.io/)             | Prisma as an ORM                                        |

---

## Project Structure

```bash
config/                                   # Used to describe the machines and fuzzy rules
├── machine1.config.json                  # Config for machine 1 with rules
├── machine2.config.json                  # Config for machine 2 with rules
├── machine3.config.json                  # Config for machine 3 with rules
prisma/                                   # All prisma
├── migrations/                           # All migrations
├── schema.prisma                         # Schema of prisma models
├── seed.ts                               # Seeder to fill database
src/
├── errors/                               # Custom Errors
│   ├── FLV_Error.ts                      # Custom FLV Error
├── mqtt-core/                            # Core MQTT logic
│   ├── mqtt.service.ts                   # Manages MQTT connection, topics, SSE streams
│   └── mqtt-debug.controller.ts          # All routes for SSE streams
│   └── mqtt-client.controller.ts         # All routes for client connection
│   └── mqtt-subscriptions.controller.ts  # All routes for subscriptions
│   └── mqtt.module.ts                    # Module for MQTT service
│   └── parse-sensor.pipe.ts              # A NestJS pipe to parse incoming MQTT sensor data
├── fuzzy-core/                           # Core Fuzzy logic
│   └── tests/                            # All the tests for the fuzzy core
│       └── FLV.spec.ts                   # FLV specific tests
│       └── MembershipFunction.spec.ts    # Membership function specific tests
│   └── FLV.ts                            # Fuzzy Linguistic Variable
│   └── fuzzy.engine.factory.ts           #
│   └── fuzzy.engine.ts                   #
│   └── fuzzy.module.ts                   #
│   └── fuzzy.service.ts                  #
│   └── fuzzy.types.ts                    #
│   └── MembershipFunctions.ts            # Membership function for ease of use for the FLV creation
├── maintenance-core/                     # Core maintenance logic
│   └── maintenance.module.ts             #
│   └── maintenance.service.ts            #
│   └── maintenance.controller.ts         #
├── navigation-core/                      # Core navigation logic
│   └── navigation.module.ts              #
│   └── navigation.service.ts             #
│   └── pathfinding.service.ts            # For holding pathfinding logic
│   └── graph.service.ts                  # For retrieving graph from database
│   └── navigation.controller.ts          #
├── app.module.ts                         # Main module
├── app.service.ts                        # Main Service
├── app.controller.ts                     # Main controller, (No endpoint defined as of for now)
├── main.ts                               # App bootstrap with Swagger setup
timeseriesdb                              # Contains SQL scripts to init and fill timeseriesdb
├── init.sql                              # Init script to create tables
└── seed.sql                              # Seeder script to fill database
```

---

## API Endpoints

Api documention using the readme website based on our swagger docs: [`View API Docs`](https://ar-mi-assistant.readme.io/reference/mqttdebugcontroller_getlatestdata#/)

### MQTT Client Management

| Method | Route                   | Description                       |
| ------ | ----------------------- | --------------------------------- |
| GET    | /mqtt/client/connect    | Connect the MQTT client           |
| DELETE | /mqtt/client/disconnect | Disconnect the MQTT client        |
| GET    | /mqtt/client/status     | Get the status of the MQTT client |

### MQTT Subscription Management

| Method | Route                      | Description                            |
| ------ | -------------------------- | -------------------------------------- |
| GET    | /mqtt/subscriptions/status | Get the status of active subscriptions |
| GET    | /mqtt/{machineId}/{sensor} | Live stream of sensor data via SSE     |
| DELETE | /mqtt/{machineId}/{sensor} | Unsubscribe from a specific machine ID |
| GET    | /mqtt/all                  | Live stream of all sensor data via SSE |
| DELETE | /mqtt/all                  | Unsubscribe from all machines          |

### MQTT Debug & Testing

| Method | Route                             | Description                                      |
| ------ | --------------------------------- | ------------------------------------------------ |
| GET    | /mqtt/latest/{machineId}/{sensor} | Get the latest message for a specific machine ID |
| GET    | /mqtt/debug/{machineId}/{sensor}  | Get live data in HTML for a specific machine ID  |

### Maintenance Evaluation

| Method | Route                          | Description                        |
| ------ | ------------------------------ | ---------------------------------- |
| GET    | /maintenance/latest/evaluation | Get the latest evaluation results  |
| GET    | /maintenance/evaluate          | Stream evaluation results via SSE  |

### Pathfinding and Navigation

| Method | Route                | Description                                                          |
| ------ | -------------------- | -------------------------------------------------------------------- |
| GET    | /navigation/graph    | Get the entire graph                                                 |
| GET    | /navigation/distance | Get the distance between to nodes determined by A\*'s shortest path  |
| GET    | /navigation/path     | Get the nodes of the shortest path determined by A\*                 |

### machine

| Method | Route               | Description                                                  |
| ------ | ------------------- | ------------------------------------------------------------ |
| GET    | /machines           | Get all machine info except the node-id's                    |
| GET    | /machines/{id}      | Get info of one machine with the id, doesnt return node id's |
| GET    | /machines/{id}/node | Get node id from a specific machine.                         |

---

## Setup Instructions

### Install dependencies

```bash
npm install
```

### Set environment variables

Create a `.env.development` file or use environment variables in your system, see `.env.example` for a template.

```bash
#remove this line and rename the file to .env.development
MQTT_HOST_IP="HOST_IP"
HOST_PORT="3000"
```

> [!TIP]
> More about the MQTT_HOST_IP is found in: [Connection to the broker](#Connection-to-the-broker)

### Environment Configuration

This project supports environment-specific configuration using `.env` files and the `NODE_ENV` variable.  
By default, the backend loads environment variables from a file based on the current `NODE_ENV`:

```bash
.env.development   # NODE_ENV=development (default)
.env.test          # NODE_ENV=test
.env.production    # NODE_ENV=production
```

To specify which environment file should be used, set `NODE_ENV`:

```bash
# Starting server in terminal
NODE_ENV=production npm run start:prod
```

```bash
# Starting server in Docker
ENV NODE_ENV=production
```

```yaml
# CI/CD pipelines
env:
  NODE_ENV: production
```

### Dependency on Relational and Timeseries database

In order to run the API succesfully you will need an instance of a timeseries database and relational database.

#### Compose File

Make sure a file called `compose.yaml` exists and it contains following content under `services`:

```yaml
relational-db:
  image: postgres:17
  container_name: AR_Assistant_RelationalDB
  env_file: .env.development
  ports:
    - '5433:5432'
  volumes:
    - AR_Assistant_RelationalDB:/var/lib/postgresql/data

timeseries-db:
  image: timescale/timescaledb:latest-pg17
  container_name: AR_Assistant_TimeseriesDB
  env_file: .env.development
  ports:
    - '5434:5432'
  volumes:
    - AR_Assistant_TimeseriesDB:/var/lib/postgresql/data
```

And this part needs to be added under `volumes`:

```yaml
AR_Assistant_RelationalDB:
  driver: local
AR_Assistant_TimeseriesDB:
  driver: local
```

#### Environment variables

Also make sure you have the following environment variables defined in the .env file:

```bash
# Database connection strings
RELATIONAL_DATABASE_URL="postgresql://<POSTGRES_USER>:<POSTGRES_PASSWORD>@<URL>:<PORT>/<POSTGRES_DB>?schema=public"
TIMESERIES_DATABASE_URL="postgresql://<POSTGRES_USER>:<POSTGRES_PASSWORD>@<URL>:<PORT>/<POSTGRES_DB>?schema=public"
POSTGRES_USER=<POSTGRES_USER>           # replace with your desired dbuser
POSTGRES_PASSWORD=<POSTGRES_PASSWORD>   # replace you your password
POSTGRES_DB=<POSTGRES_DB>               # set this to the name of the desired db name
```

> [!WARNING]
> Make sure the port in the connection string is the same as the left side of the port forwarding in you docker compose for that database

> [!TIP]
> For more information about setting these environment variables checkout: [Set environment variables](#Set-environment-variables).

### Dependency on MQTT broker and MQTT sensors

In order to run the API succesfully you will need a MQTT Broker and some sensors. Luckily We have though about that and will describe it below how to set that up.  
The MQTT broker en sensor are deployed using docker compose, if you do not have this please follow the instructions described [here](https://docs.docker.com/compose/install/).

#### Compose File

Make sure a file called `compose.yaml` exists and it contains following content under `services`

```yaml
mqtt5:
  image: eclipse-mosquitto
  container_name: mqtt5
  ports:
    - '1883:1883' # default mqtt port
    - '9001:9001' # optional port for websocket support
  volumes:
    - ./mosquitto/config:/mosquitto/config:rw
    - ./mosquitto/data:/mosquitto/data:rw
    - ./mosquitto/log:/mosquitto/log:rw
  restart: unless-stopped

TemperatureSensor:
  image: harbor.casperschouwenaar.nl/arassistant/realtime_sensor:dev
  container_name: TemperatureSensor
  environment:
    - PORT=3000
    - MQTT_URL=mqtt://mqtt5:1883
    - MQTT_TOPIC=temperature
    - MACHINE_ID=1
  depends_on:
    - mqtt5

PressureSensor:
  image: harbor.casperschouwenaar.nl/arassistant/realtime_sensor:dev
  container_name: PressureSensor
  environment:
    - PORT=3000
    - MQTT_URL=mqtt://mqtt5:1883
    - MQTT_TOPIC=pressure
    - MACHINE_ID=1
  depends_on:
    - mqtt5

TemperatureSensor2:
  image: harbor.casperschouwenaar.nl/arassistant/realtime_sensor:dev
  container_name: TemperatureSensor2
  environment:
    - PORT=3000
    - MQTT_URL=mqtt://mqtt5:1883
    - MQTT_TOPIC=temperature
    - MACHINE_ID=2
  depends_on:
    - mqtt5

PressureSensor2:
  image: harbor.casperschouwenaar.nl/arassistant/realtime_sensor:dev
  container_name: PressureSensor2
  environment:
    - PORT=3000
    - MQTT_URL=mqtt://mqtt5:1883
    - MQTT_TOPIC=pressure
    - MACHINE_ID=2
  depends_on:
    - mqtt5

VibrationSensor2:
  image: harbor.casperschouwenaar.nl/arassistant/realtime_sensor:dev
  container_name: VibrationSensor2
  environment:
    - PORT=3000
    - MQTT_URL=mqtt://mqtt5:1883
    - MQTT_TOPIC=vibration
    - MACHINE_ID=2
  depends_on:
    - mqtt5

VibrationSensor3:
  image: harbor.casperschouwenaar.nl/arassistant/realtime_sensor:dev
  container_name: VibrationSensor3
  environment:
    - PORT=3000
    - MQTT_URL=mqtt://mqtt5:1883
    - MQTT_TOPIC=vibration
    - MACHINE_ID=3
  depends_on:
    - mqtt5
```

And this part needs to be added under `networks`:

```yaml
default:
  name: mqtt5-network
```

#### Extra explanation about the mosquitto config

In order to run mosquitto it needs a config file. for this you make the following file structure:

```bash
compose.yaml
mosquitto/
├── config/
│   ├── mosquitto.conf
├── data/
├── log/
```

Within the `mosquitto.conf` file add teh following content:

```conf
# mosquitto.conf
persistence true
persistence_location /mosquitto/data/
log_dest file /mosquitto/log/mosquitto.log

# Default listener (non-TLS)
listener 1883
protocol mqtt

# WebSocket listener
listener 9001
protocol websockets

allow_anonymous true
```

#### Connection to the broker

The api need the broker to function this is done by using the MQTT_HOST_IP environment variable.  
Because we used the docker compose and used a port forwarding on the broker we can set the `MQTT_HOST_IP` to a value of `localhost`

> [!TIP]
> More on how to set this is found in chapter: [Set environment variables](#Set-environment-variables).

### Starting the dependencies (MQTT, sensors and DB's)

There are two commands you should know:

- `docker compose up -d` this is used to start the broker and the sensors
- `docker compose down` is used to stop the broker and sensors

An addition command you could you is:

- `docker compose up` is similar to the starting of the broker and the sensor with the only difference that you run them in the foreground and imediatly stop them when exiting using `Ctrl-C`.

### Database preperations

```bash
# Prisma (Relational)
npm run prisma:generate
npm run prisma:migrate
npm run prisma:seed

# TimeseriesDb
```

> [!CAUTION]
> TimeseriesDB initialization and seeding not implemented yet.

### Start the project

```bash
# Development with hot-reload
npm run start:dev

# Production build
npm run start:prod
```

---

## Features

- Real-time data updates via **Server-Sent Events**
- Dynamic topic subscription/unsubscription per machine
- Automatic reconnection + resubscription after MQTT disconnects
- Latest message snapshot endpoints for polling clients
- Swagger documentation included
- Fuzzy Logic-Based Maintenance Detection

---

## Planned Features

These features are part of the product vision and will be implemented in future development stages:

- **Sensor Data Persistence**  
  Real-time sensor data will be saved to a database for historical analysis and offline insights.

---

## Testing

```bash
# Unit tests
npm run test

# Unit tests with coverage
npm run test:dev

# Unit tests with watching
npm run test:watch
```

---

## Deployment

This README file can be used in order to host the complete system locally.  
We have this exact configuration also running at https://api.arassistant.nl.

> [!Note]
> For a more detailed guide about a NestJS deployment please checkout [the offical guide](https://docs.nestjs.com/deployment)

---

## Project Info

- Project by a Software Engineering students at **Windesheim Zwolle**
- Coach: **...**
- Unity frontend consumes this API for AR machine overlays
- Built for the module `ICT.GP.PRJCT.V22_2425`

## Authors

- ...
- ...
- ...
- ...

---

## License

This project is for educational use and not licensed for commercial use.
